'use client'

import { useEffect, useState, useRef } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Pago, Cliente, Suscripcion } from '@/lib/types'
import Link from 'next/link'
import { format, addMonths, addYears } from 'date-fns'
import jsPDF from 'jspdf'
import html2canvas from 'html2canvas'
import Modal from '../../components/Modal'

export default function PagoDetailPage() {
  const params = useParams()
  const router = useRouter()
  const [pago, setPago] = useState<Pago | null>(null)
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [suscripcion, setSuscripcion] = useState<Suscripcion | null>(null)
  const [loading, setLoading] = useState(true)
  const [marcandoCompletado, setMarcandoCompletado] = useState(false)
  const [eliminando, setEliminando] = useState(false)
  const [generandoPDF, setGenerandoPDF] = useState(false)
  const [mostrandoMensajeWhatsApp, setMostrandoMensajeWhatsApp] = useState(false)
  const comprobanteRef = useRef<HTMLDivElement>(null)
  const [modalConfirmarCompletar, setModalConfirmarCompletar] = useState(false)
  const [modalConfirmarEliminar, setModalConfirmarEliminar] = useState(false)
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' | 'warning' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    if (params.id) {
      fetchPago(params.id as string)
    }
  }, [params.id])

  async function fetchPago(id: string) {
    try {
      const { data, error } = await supabase
        .from('pagos')
        .select(`
          *,
          cliente:clientes(*),
          suscripcion:suscripciones(*)
        `)
        .eq('id', id)
        .single()

      if (error) throw error
      setPago(data)
      setCliente(data.cliente as Cliente)
      setSuscripcion(data.suscripcion as Suscripcion | null)
    } catch (error) {
      console.error('Error fetching pago:', error)
    } finally {
      setLoading(false)
    }
  }

  function abrirModalCompletar() {
    if (!pago || pago.estado === 'completado') return
    setModalConfirmarCompletar(true)
  }

  async function marcarComoCompletado() {
    if (!pago || pago.estado === 'completado') return

    setMarcandoCompletado(true)
    setModalConfirmarCompletar(false)

    try {
      // 1. Marcar el pago actual como completado
      const { error: updateError } = await supabase
        .from('pagos')
        .update({ estado: 'completado' })
        .eq('id', pago.id)

      if (updateError) throw updateError

      // 2. Si hay una suscripción asociada, crear el siguiente pago recurrente (siempre, sin importar fecha fin)
      let siguientePagoCreado = false
      if (pago.suscripcion_id && suscripcion) {
        // Usar fecha_fin del pago actual, o calcular desde fecha_inicio, o usar fecha_pago
        const fechaBaseStr = pago.fecha_fin || pago.fecha_inicio || pago.fecha_pago
        const [year, month, day] = fechaBaseStr.split('T')[0].split('-').map(Number)
        
        let siguienteYear = year
        let siguienteMonth = month
        let siguienteDay = day

        if (suscripcion.tipo === 'anual') {
          siguienteYear = year + 1
        } else {
          siguienteMonth = month + 1
          if (siguienteMonth > 12) {
            siguienteMonth = 1
            siguienteYear = year + 1
          }
        }

        const siguienteFechaInicio = `${siguienteYear}-${siguienteMonth.toString().padStart(2, '0')}-${siguienteDay.toString().padStart(2, '0')}`
        
        // Calcular fecha fin del siguiente pago
        let siguienteFechaFin: string
        if (suscripcion.tipo === 'anual') {
          siguienteFechaFin = `${siguienteYear + 1}-${siguienteMonth.toString().padStart(2, '0')}-${siguienteDay.toString().padStart(2, '0')}`
        } else {
          let finMonth = siguienteMonth + 1
          let finYear = siguienteYear
          if (finMonth > 12) {
            finMonth = 1
            finYear = siguienteYear + 1
          }
          siguienteFechaFin = `${finYear}-${finMonth.toString().padStart(2, '0')}-${siguienteDay.toString().padStart(2, '0')}`
        }

          // Crear siempre el siguiente pago recurrente (sin verificar fecha fin de suscripción)
          const { data: nuevoPagoData, error: nuevoPagoError } = await supabase
            .from('pagos')
            .insert([{
              cliente_id: pago.cliente_id,
              suscripcion_id: pago.suscripcion_id,
              monto: suscripcion.precio,
              fecha_pago: siguienteFechaInicio,
              fecha_inicio: siguienteFechaInicio,
              fecha_fin: siguienteFechaFin,
              estado: 'pendiente',
              notas: `Pago ${suscripcion.tipo === 'anual' ? 'anual' : 'mensual'} recurrente - Generado automáticamente`,
            }])
            .select()

          if (nuevoPagoError) {
            console.error('Error creating siguiente pago:', nuevoPagoError)
            setModalMensaje({
              isOpen: true,
              mensaje: `Error al crear el siguiente pago recurrente: ${nuevoPagoError.message}`,
              tipo: 'error'
            })
          } else if (nuevoPagoData && nuevoPagoData.length > 0) {
            siguientePagoCreado = true
            console.log('✅ Siguiente pago creado exitosamente:', nuevoPagoData[0])
          }
      }

      // Recargar los datos
      await fetchPago(pago.id)
      
      if (siguientePagoCreado) {
        setModalMensaje({
          isOpen: true,
          mensaje: '✅ Pago marcado como completado. Se creó el siguiente pago recurrente automáticamente.',
          tipo: 'success'
        })
      } else if (pago.suscripcion_id && suscripcion) {
        setModalMensaje({
          isOpen: true,
          mensaje: '⚠️ Pago marcado como completado, pero no se pudo crear el siguiente pago recurrente. Verifica la consola para más detalles.',
          tipo: 'warning'
        })
      } else {
        setModalMensaje({
          isOpen: true,
          mensaje: '✅ Pago marcado como completado exitosamente.',
          tipo: 'success'
        })
      }
      
      setTimeout(() => {
        router.push('/pagos')
      }, 1500)
    } catch (error: any) {
      console.error('Error updating pago:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al marcar el pago: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setMarcandoCompletado(false)
    }
  }

  function abrirModalEliminar() {
    setModalConfirmarEliminar(true)
  }

  async function eliminarPago() {
    if (!pago) return

    setEliminando(true)
    setModalConfirmarEliminar(false)

    try {
      const { error } = await supabase
        .from('pagos')
        .delete()
        .eq('id', pago.id)

      if (error) throw error

      setModalMensaje({
        isOpen: true,
        mensaje: 'Pago eliminado exitosamente',
        tipo: 'success'
      })
      setTimeout(() => {
        router.push('/pagos')
      }, 1500)
    } catch (error: any) {
      console.error('Error deleting pago:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al eliminar el pago: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setEliminando(false)
    }
  }

  async function descargarPDF() {
    if (!comprobanteRef.current || !pago || !cliente) return

    setGenerandoPDF(true)

    try {
      const canvas = await html2canvas(comprobanteRef.current, {
        scale: 2,
        useCORS: true,
        logging: false,
      })

      const imgData = canvas.toDataURL('image/png')
      const pdf = new jsPDF('p', 'mm', 'a4')
      const pdfWidth = pdf.internal.pageSize.getWidth()
      const pdfHeight = pdf.internal.pageSize.getHeight()
      const imgWidth = canvas.width
      const imgHeight = canvas.height
      const ratio = Math.min(pdfWidth / imgWidth, pdfHeight / imgHeight)
      const imgX = (pdfWidth - imgWidth * ratio) / 2
      const imgY = 0

      pdf.addImage(imgData, 'PNG', imgX, imgY, imgWidth * ratio, imgHeight * ratio)
      pdf.save(`comprobante-pago-${pago.id.slice(0, 8)}.pdf`)
    } catch (error) {
      console.error('Error generating PDF:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: 'Error al generar el PDF',
        tipo: 'error'
      })
    } finally {
      setGenerandoPDF(false)
    }
  }

  function obtenerURLComprobante() {
    if (typeof window === 'undefined') return ''
    return `${window.location.origin}/pagos/${pago?.id}/comprobante`
  }

  function generarMensajeWhatsApp() {
    if (!pago || !cliente) return ''

    const fechaPago = pago.fecha_pago.split('T')[0]
    const [year, month, day] = fechaPago.split('-')
    const fechaFormateada = `${day}/${month}/${year}`
    const urlComprobante = obtenerURLComprobante()

    const mensaje = `Hola ${cliente.nombre || cliente.empresa || 'Cliente'} 👋

Te envío el comprobante de pago:

📋 *Detalles del Pago:*
• Cliente: ${cliente.nombre || cliente.empresa || 'Cliente'}${cliente.nombre && cliente.empresa ? ` - ${cliente.empresa}` : ''}
• Monto: $${pago.monto.toLocaleString()}
• Fecha: ${fechaFormateada}
• Método de pago: ${pago.metodo_pago || 'No especificado'}
${pago.referencia ? `• Referencia: ${pago.referencia}` : ''}
• Estado: ${pago.estado === 'completado' ? '✅ Completado' : pago.estado}

${pago.notas ? `Notas: ${pago.notas}\n` : ''}🔗 *Ver comprobante completo:*
${urlComprobante}

Gracias por tu pago. Si tienes alguna consulta, no dudes en contactarme.

Saludos cordiales,
NAVIGPS`

    return mensaje
  }

  function copiarMensajeWhatsApp() {
    const mensaje = generarMensajeWhatsApp()
    navigator.clipboard.writeText(mensaje).then(() => {
      setModalMensaje({
        isOpen: true,
        mensaje: '✅ Mensaje copiado al portapapeles. Ahora puedes pegarlo en WhatsApp.',
        tipo: 'success'
      })
    }).catch(() => {
      setModalMensaje({
        isOpen: true,
        mensaje: 'Error al copiar el mensaje. Por favor, cópialo manualmente.',
        tipo: 'error'
      })
    })
  }

  function abrirWhatsApp() {
    const mensaje = generarMensajeWhatsApp()
    const mensajeCodificado = encodeURIComponent(mensaje)
    const telefono = cliente?.telefono?.replace(/\D/g, '') || ''
    
    if (telefono) {
      window.open(`https://wa.me/${telefono}?text=${mensajeCodificado}`, '_blank')
    } else {
      // Si no hay teléfono, solo mostrar el mensaje para copiar
      setMostrandoMensajeWhatsApp(true)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  if (!pago) {
    return (
      <div className="text-center p-8">
        <p>Pago no encontrado</p>
        <Link href="/pagos" className="text-blue-600 hover:underline">Volver a pagos</Link>
      </div>
    )
  }

  // Trabajar con strings YYYY-MM-DD directamente para evitar problemas de zona horaria
  const fechaPagoStr = pago.fecha_pago.split('T')[0]
  const hoyStr = new Date().toISOString().split('T')[0]
  const vencido = fechaPagoStr < hoyStr
  
  const porVencer = (() => {
    const [year, month, day] = fechaPagoStr.split('-').map(Number)
    const [yearHoy, monthHoy, dayHoy] = hoyStr.split('-').map(Number)
    const fecha = new Date(year, month - 1, day)
    const hoy = new Date(yearHoy, monthHoy - 1, dayHoy)
    const diasRestantes = Math.ceil((fecha.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
    return diasRestantes <= 7 && diasRestantes > 0 && fechaPagoStr >= hoyStr
  })()

  return (
    <div className="max-w-4xl mx-auto">
      <div className="mb-4 flex justify-between items-center">
        <Link href="/pagos" className="text-blue-600 hover:underline">← Volver a pagos</Link>
        <div className="flex gap-2">
          <button
            onClick={() => setMostrandoMensajeWhatsApp(true)}
            className="bg-green-500 text-white px-4 py-2 rounded hover:bg-green-600 transition text-sm flex items-center gap-2"
          >
            💬 WhatsApp
          </button>
          <button
            onClick={descargarPDF}
            disabled={generandoPDF || !pago}
            className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50 transition text-sm"
          >
            {generandoPDF ? 'Generando...' : '📄 Descargar PDF'}
          </button>
          <Link
            href={`/pagos/${pago?.id}/editar`}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition text-sm"
          >
            ✏️ Editar
          </Link>
          <button
            onClick={eliminarPago}
            disabled={eliminando}
            className="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700 disabled:opacity-50 transition text-sm"
          >
            {eliminando ? 'Eliminando...' : '🗑️ Eliminar'}
          </button>
        </div>
      </div>
      
      <div className={`bg-white p-6 rounded-lg shadow mb-6 ${
        pago.estado === 'completado' ? 'border-l-4 border-green-500' :
        vencido ? 'border-l-4 border-red-500' :
        porVencer ? 'border-l-4 border-yellow-500' :
        'border-l-4 border-blue-500'
      }`}>
        <div className="flex justify-between items-start mb-6">
          <div>
            <h1 className="text-3xl font-bold mb-2">Pago #{pago.id.slice(0, 8)}</h1>
            <Link href={`/clientes/${pago.cliente_id}`} className="text-blue-600 hover:underline">
              👤 {cliente?.nombre || 'Cliente desconocido'}
            </Link>
          </div>
          <span className={`px-3 py-1 rounded text-sm font-medium ${
            pago.estado === 'completado' ? 'bg-green-100 text-green-800' :
            pago.estado === 'cancelado' ? 'bg-gray-100 text-gray-800' :
            vencido ? 'bg-red-100 text-red-800' :
            porVencer ? 'bg-yellow-100 text-yellow-800' :
            'bg-blue-100 text-blue-800'
          }`}>
            {pago.estado === 'completado' ? 'Completado' :
             pago.estado === 'cancelado' ? 'Cancelado' :
             vencido ? 'Vencido' :
             porVencer ? 'Por vencer' :
             'En curso'}
          </span>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Monto</h3>
            <p className="text-2xl font-bold text-gray-900">${pago.monto.toLocaleString()}</p>
          </div>
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Fecha de Pago</h3>
            <p className="text-gray-900">
              {(() => {
                const fecha = pago.fecha_pago.split('T')[0]
                const [year, month, day] = fecha.split('-')
                return `${day}/${month}/${year}`
              })()}
            </p>
            {pago.fecha_inicio && (
              <div className="mt-2">
                <h3 className="font-semibold text-gray-700 mb-1">Fecha de Inicio</h3>
                <p className="text-gray-900">
                  {(() => {
                    const fecha = pago.fecha_inicio.split('T')[0]
                    const [year, month, day] = fecha.split('-')
                    return `${day}/${month}/${year}`
                  })()}
                </p>
              </div>
            )}
            {pago.fecha_fin && (
              <div className="mt-2">
                <h3 className="font-semibold text-gray-700 mb-1">Fecha de Fin</h3>
                <p className="text-gray-900">
                  {(() => {
                    const fecha = pago.fecha_fin.split('T')[0]
                    const [year, month, day] = fecha.split('-')
                    return `${day}/${month}/${year}`
                  })()}
                </p>
              </div>
            )}
          </div>
          {pago.metodo_pago && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Método de Pago</h3>
              <p className="text-gray-900 capitalize">{pago.metodo_pago}</p>
            </div>
          )}
          {pago.referencia && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Referencia</h3>
              <p className="text-gray-900">{pago.referencia}</p>
            </div>
          )}
          {suscripcion && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Suscripción</h3>
              <Link href={`/suscripciones/${suscripcion.id}`} className="text-blue-600 hover:underline">
                {suscripcion.tipo === 'anual' ? 'Anual' : 'Mensual'} - ${suscripcion.precio.toLocaleString()}
              </Link>
            </div>
          )}
          {pago.notas && (
            <div className="md:col-span-2">
              <h3 className="font-semibold text-gray-700 mb-1">Notas</h3>
              <p className="text-gray-900 whitespace-pre-wrap">{pago.notas}</p>
            </div>
          )}
        </div>

        {pago.estado === 'pendiente' && (
          <div className="mt-6 pt-6 border-t">
            <button
              onClick={abrirModalCompletar}
              disabled={marcandoCompletado}
              className="bg-green-600 text-white px-6 py-2 rounded hover:bg-green-700 disabled:opacity-50 transition font-medium"
            >
              {marcandoCompletado ? 'Procesando...' : '✓ Marcar como Completado'}
            </button>
            <p className="text-sm text-gray-600 mt-2">
              {suscripcion 
                ? 'Al marcar como completado, se creará automáticamente el siguiente pago pendiente.'
                : 'Este pago no tiene suscripción asociada, solo se marcará como completado.'}
            </p>
          </div>
        )}
      </div>

      {/* Comprobante Visual */}
      <div className="bg-white p-6 rounded-lg shadow">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold">Comprobante de Pago</h2>
          <div className="flex gap-2">
            <button
              onClick={() => {
                const url = obtenerURLComprobante()
                navigator.clipboard.writeText(url)
                setModalMensaje({
                  isOpen: true,
                  mensaje: 'URL del comprobante copiada al portapapeles',
                  tipo: 'success'
                })
              }}
              className="bg-gray-600 text-white px-4 py-2 rounded hover:bg-gray-700 transition text-sm"
            >
              📋 Copiar URL
            </button>
            <button
              onClick={descargarPDF}
              disabled={generandoPDF}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50 transition text-sm"
            >
              {generandoPDF ? 'Generando...' : '📄 Descargar PDF'}
            </button>
          </div>
        </div>

        <div ref={comprobanteRef} className="bg-white border-2 border-gray-300 p-8 max-w-2xl mx-auto" style={{ minHeight: '400px' }}>
          <div className="text-center mb-8">
            <h1 className="text-3xl font-bold text-gray-800 mb-2">NAVIGPS</h1>
            <p className="text-gray-600">Comprobante de Pago</p>
          </div>

          <div className="border-t-2 border-b-2 border-gray-400 py-4 mb-6">
            <div className="flex justify-between items-center mb-2">
              <span className="text-gray-600">Número de Comprobante:</span>
              <span className="font-bold text-gray-800">#{pago.id.slice(0, 8).toUpperCase()}</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Fecha de Emisión:</span>
              <span className="font-semibold text-gray-800">{format(new Date(), 'dd/MM/yyyy HH:mm')}</span>
            </div>
          </div>

          <div className="mb-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-3">Datos del Cliente</h3>
            <div className="space-y-2 text-gray-700">
              <p><span className="font-medium">Nombre:</span> {cliente?.nombre || 'N/A'}</p>
              {cliente?.empresa && <p><span className="font-medium">Empresa:</span> {cliente.empresa}</p>}
              {cliente?.email && <p><span className="font-medium">Email:</span> {cliente.email}</p>}
              {cliente?.telefono && <p><span className="font-medium">Teléfono:</span> {cliente.telefono}</p>}
              {cliente?.rfc && <p><span className="font-medium">RUT:</span> {cliente.rfc}</p>}
            </div>
          </div>

          <div className="mb-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-3">Detalles del Pago</h3>
            <div className="bg-gray-50 p-4 rounded space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-700">Monto:</span>
                <span className="text-2xl font-bold text-gray-900">${pago.monto.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-700">Fecha de Pago:</span>
                <span className="font-semibold text-gray-800">
                  {(() => {
                    const fecha = pago.fecha_pago.split('T')[0]
                    const [year, month, day] = fecha.split('-')
                    return `${day}/${month}/${year}`
                  })()}
                </span>
                {pago.fecha_inicio && (
                  <>
                    <span className="text-gray-700">Fecha Inicio:</span>
                    <span className="font-semibold text-gray-800">
                      {(() => {
                        const fecha = pago.fecha_inicio.split('T')[0]
                        const [year, month, day] = fecha.split('-')
                        return `${day}/${month}/${year}`
                      })()}
                    </span>
                  </>
                )}
                {pago.fecha_fin && (
                  <>
                    <span className="text-gray-700">Fecha Fin:</span>
                    <span className="font-semibold text-gray-800">
                      {(() => {
                        const fecha = pago.fecha_fin.split('T')[0]
                        const [year, month, day] = fecha.split('-')
                        return `${day}/${month}/${year}`
                      })()}
                    </span>
                  </>
                )}
              </div>
              {pago.metodo_pago && (
                <div className="flex justify-between">
                  <span className="text-gray-700">Método de Pago:</span>
                  <span className="font-semibold text-gray-800 capitalize">{pago.metodo_pago}</span>
                </div>
              )}
              {pago.referencia && (
                <div className="flex justify-between">
                  <span className="text-gray-700">Referencia:</span>
                  <span className="font-semibold text-gray-800">{pago.referencia}</span>
                </div>
              )}
              <div className="flex justify-between">
                <span className="text-gray-700">Estado:</span>
                <span className={`font-semibold ${
                  pago.estado === 'completado' ? 'text-green-600' :
                  pago.estado === 'pendiente' ? 'text-yellow-600' :
                  'text-red-600'
                }`}>
                  {pago.estado === 'completado' ? 'Completado' :
                   pago.estado === 'pendiente' ? 'Pendiente' :
                   'Cancelado'}
                </span>
              </div>
              {suscripcion && (
                <div className="flex justify-between">
                  <span className="text-gray-700">Suscripción:</span>
                  <span className="font-semibold text-gray-800">
                    {suscripcion.tipo === 'anual' ? 'Anual' : 'Mensual'} - ${suscripcion.precio.toLocaleString()}
                  </span>
                </div>
              )}
            </div>
          </div>

          {pago.notas && (
            <div className="mb-6">
              <h3 className="text-lg font-semibold text-gray-800 mb-2">Notas</h3>
              <p className="text-gray-700">{pago.notas}</p>
            </div>
          )}

          <div className="border-t-2 border-gray-400 pt-4 mt-6">
            <p className="text-sm text-gray-600 text-center">
              Este es un comprobante generado automáticamente por el sistema NAVIGPS
            </p>
            <p className="text-xs text-gray-500 text-center mt-2">
              URL: {obtenerURLComprobante()}
            </p>
          </div>
        </div>
      </div>

      {/* Modal para mensaje de WhatsApp */}
      {mostrandoMensajeWhatsApp && (
        <div className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="p-6">
              <div className="flex justify-between items-center mb-4">
                <h2 className="text-2xl font-bold">Mensaje para WhatsApp</h2>
                <button
                  onClick={() => setMostrandoMensajeWhatsApp(false)}
                  className="text-gray-500 hover:text-gray-700 text-2xl"
                >
                  ×
                </button>
              </div>
              
              <div className="mb-4">
                <label className="block text-sm font-medium mb-2">Mensaje predefinido:</label>
                <textarea
                  readOnly
                  value={generarMensajeWhatsApp()}
                  className="w-full border rounded-lg p-4 h-64 font-mono text-sm"
                />
              </div>

              <div className="flex gap-3">
                <button
                  onClick={copiarMensajeWhatsApp}
                  className="flex-1 bg-green-500 text-white px-4 py-2 rounded hover:bg-green-600 transition"
                >
                  📋 Copiar Mensaje
                </button>
                {cliente?.telefono && (
                  <button
                    onClick={() => {
                      abrirWhatsApp()
                      setMostrandoMensajeWhatsApp(false)
                    }}
                    className="flex-1 bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 transition"
                  >
                    💬 Abrir WhatsApp
                  </button>
                )}
                <button
                  onClick={() => setMostrandoMensajeWhatsApp(false)}
                  className="px-4 py-2 border rounded hover:bg-gray-50 transition"
                >
                  Cerrar
                </button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal de confirmación para completar pago */}
      <Modal
        isOpen={modalConfirmarCompletar}
        onClose={() => setModalConfirmarCompletar(false)}
        title="Completar Pago"
        message="¿Marcar este pago como completado? Esto creará automáticamente el siguiente pago pendiente si hay una suscripción asociada."
        onConfirm={marcarComoCompletado}
        confirmText="Completar"
        cancelText="Cancelar"
        type="info"
      />

      {/* Modal de confirmación para eliminar */}
      <Modal
        isOpen={modalConfirmarEliminar}
        onClose={() => setModalConfirmarEliminar(false)}
        title="Eliminar Pago"
        message="¿Estás seguro de eliminar este pago? Esta acción no se puede deshacer."
        onConfirm={eliminarPago}
        confirmText="Eliminar"
        cancelText="Cancelar"
        type="warning"
      />

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : modalMensaje.tipo === 'error' ? 'Error' : 'Advertencia'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
